import { useState, useEffect } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { toast } from 'sonner';
import { Cake, ArrowLeft, Search, Package, CheckCircle, Clock, XCircle } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const TrackOrderPage = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [orderId, setOrderId] = useState(searchParams.get('id') || '');
  const [order, setOrder] = useState(null);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    const id = searchParams.get('id');
    if (id) {
      setOrderId(id);
      handleTrack(id);
    }
  }, []);

  const handleTrack = async (id = orderId) => {
    if (!id) {
      toast.error('Please enter an order ID');
      return;
    }

    setLoading(true);
    try {
      const response = await axios.get(`${API}/orders/track/${id}`);
      setOrder(response.data);
      toast.success('Order found!');
    } catch (error) {
      console.error('Error tracking order:', error);
      toast.error(error.response?.status === 404 ? 'Order not found' : 'Failed to track order');
      setOrder(null);
    } finally {
      setLoading(false);
    }
  };

  const getStatusIcon = (status) => {
    switch (status) {
      case 'pending':
        return <Clock className="w-6 h-6 text-yellow-500" />;
      case 'confirmed':
        return <CheckCircle className="w-6 h-6 text-blue-500" />;
      case 'in_progress':
        return <Package className="w-6 h-6 text-purple-500" />;
      case 'completed':
        return <CheckCircle className="w-6 h-6 text-green-500" />;
      case 'cancelled':
        return <XCircle className="w-6 h-6 text-red-500" />;
      default:
        return <Clock className="w-6 h-6 text-gray-500" />;
    }
  };

  const getStatusText = (status) => {
    switch (status) {
      case 'pending':
        return 'Order Received';
      case 'confirmed':
        return 'Order Confirmed';
      case 'in_progress':
        return 'Baking in Progress';
      case 'completed':
        return 'Order Completed';
      case 'cancelled':
        return 'Order Cancelled';
      default:
        return status;
    }
  };

  return (
    <div className="min-h-screen py-8 px-4 sm:px-6 lg:px-8">
      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <Button
            data-testid="back-to-home-btn"
            variant="ghost"
            onClick={() => navigate('/')}
            className="mb-4 text-pink-600 hover:text-pink-700 hover:bg-pink-50"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Home
          </Button>
          
          <div className="flex items-center space-x-3 mb-2">
            <Cake className="w-10 h-10 text-pink-500" />
            <h1 className="text-3xl sm:text-4xl font-bold text-gray-800">Track Your Order</h1>
          </div>
          <p className="text-gray-600">Enter your order ID to check the status</p>
        </div>

        {/* Search Form */}
        <div className="glass rounded-3xl p-6 sm:p-8 mb-8">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <Label htmlFor="orderId" className="text-gray-700 mb-2 block">Order ID</Label>
              <Input
                data-testid="order-id-input"
                id="orderId"
                value={orderId}
                onChange={(e) => setOrderId(e.target.value)}
                placeholder="Enter your order ID"
                className="w-full"
                onKeyPress={(e) => e.key === 'Enter' && handleTrack()}
              />
            </div>
            <div className="sm:self-end">
              <Button
                data-testid="track-order-btn"
                onClick={() => handleTrack()}
                disabled={loading}
                className="bg-pink-500 hover:bg-pink-600 text-white px-8 py-6 rounded-full w-full sm:w-auto"
              >
                <Search className="w-5 h-5 mr-2" />
                {loading ? 'Tracking...' : 'Track Order'}
              </Button>
            </div>
          </div>
        </div>

        {/* Order Details */}
        {order && (
          <div className="space-y-6">
            {/* Status Card */}
            <div className="glass rounded-3xl p-6 sm:p-8">
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-2xl font-semibold text-gray-800">Order Status</h2>
                <div data-testid="order-status-badge" className="flex items-center space-x-2 px-4 py-2 bg-pink-100 rounded-full">
                  {getStatusIcon(order.status)}
                  <span className="font-semibold text-gray-700">{getStatusText(order.status)}</span>
                </div>
              </div>

              {/* Timeline */}
              <div className="space-y-4">
                {['pending', 'confirmed', 'in_progress', 'completed'].map((status, index) => (
                  <div key={status} className="flex items-start space-x-4">
                    <div className={`flex flex-col items-center ${
                      index < ['pending', 'confirmed', 'in_progress', 'completed'].indexOf(order.status) ? 'opacity-100' : 'opacity-30'
                    }`}>
                      {getStatusIcon(status)}
                      {index < 3 && (
                        <div className="w-0.5 h-12 bg-pink-200 mt-2" />
                      )}
                    </div>
                    <div className="flex-1">
                      <h3 className="font-semibold text-gray-800">{getStatusText(status)}</h3>
                      <p className="text-sm text-gray-600">
                        {status === 'pending' && 'We received your order'}
                        {status === 'confirmed' && 'Your order has been confirmed'}
                        {status === 'in_progress' && 'Your cake is being prepared'}
                        {status === 'completed' && 'Your order is ready'}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Customer Details */}
            <div className="glass rounded-3xl p-6 sm:p-8">
              <h2 className="text-2xl font-semibold text-gray-800 mb-6">Order Details</h2>
              <div className="grid sm:grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-gray-600">Order ID</p>
                  <p data-testid="order-id-display" className="font-semibold text-gray-800">{order.id}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Customer Name</p>
                  <p data-testid="customer-name-display" className="font-semibold text-gray-800">{order.customer_name}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Email</p>
                  <p className="font-semibold text-gray-800">{order.email}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Phone</p>
                  <p className="font-semibold text-gray-800">{order.phone}</p>
                </div>
                <div className="sm:col-span-2">
                  <p className="text-sm text-gray-600">Delivery Address</p>
                  <p className="font-semibold text-gray-800">{order.delivery_address}</p>
                </div>
              </div>
            </div>

            {/* Cake Details */}
            <div className="glass rounded-3xl p-6 sm:p-8">
              <h2 className="text-2xl font-semibold text-gray-800 mb-6">Cake Details</h2>
              <div className="grid sm:grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-gray-600">Budget</p>
                  <p className="font-semibold text-gray-800">₦{order.budget.toLocaleString()}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Cake Size</p>
                  <p className="font-semibold text-gray-800 capitalize">{order.cake_size}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Servings</p>
                  <p className="font-semibold text-gray-800">{order.servings} people</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Design Method</p>
                  <p className="font-semibold text-gray-800 capitalize">{order.design_type}</p>
                </div>
                {order.design_description && (
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600">Design Description</p>
                    <p className="font-semibold text-gray-800">{order.design_description}</p>
                  </div>
                )}
                {order.design_image && (
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600 mb-2">Design Image</p>
                    <img
                      src={order.design_image}
                      alt="Design"
                      className="w-full max-w-md rounded-lg shadow-md"
                    />
                  </div>
                )}
                {order.cake_message && (
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600">Cake Message</p>
                    <p className="font-semibold text-gray-800">{order.cake_message}</p>
                  </div>
                )}
                {order.special_requests && (
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600">Special Requests</p>
                    <p className="font-semibold text-gray-800">{order.special_requests}</p>
                  </div>
                )}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default TrackOrderPage;