from fastapi import FastAPI, APIRouter, HTTPException
from fastapi.responses import JSONResponse
from dotenv import load_dotenv
from starlette.middleware.cors import CORSMiddleware
from motor.motor_asyncio import AsyncIOMotorClient
import os
import logging
from pathlib import Path
from pydantic import BaseModel, Field, ConfigDict, EmailStr
from typing import List, Optional
import uuid
from datetime import datetime, timezone
import smtplib
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
import ssl
import random
import string

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

# Setup logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Function to generate unique tracking code
def generate_tracking_code():
    """Generate a unique 8-character tracking code like CC-XXXX-XXXX"""
    part1 = ''.join(random.choices(string.ascii_uppercase + string.digits, k=4))
    part2 = ''.join(random.choices(string.ascii_uppercase + string.digits, k=4))
    return f"CC-{part1}-{part2}"

# MongoDB connection
mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

# Email configuration
SMTP_SERVER = os.environ.get('SMTP_SERVER')
SMTP_PORT = int(os.environ.get('SMTP_PORT', 465))
SMTP_EMAIL = os.environ.get('SMTP_EMAIL')
SMTP_PASSWORD = os.environ.get('SMTP_PASSWORD')
NOTIFICATION_EMAIL = os.environ.get('NOTIFICATION_EMAIL')

app = FastAPI()
api_router = APIRouter(prefix="/api")

# Email sending function
def send_order_notification(order_data: dict):
    """Send email notification when a new order is placed"""
    try:
        # Create message
        msg = MIMEMultipart('alternative')
        msg['Subject'] = f"New Cake Order from {order_data['customer_name']}"
        msg['From'] = SMTP_EMAIL
        msg['To'] = NOTIFICATION_EMAIL

        # Create HTML email body
        html_body = f"""
        <html>
        <head>
            <style>
                body {{ font-family: Arial, sans-serif; line-height: 1.6; color: #333; }}
                .container {{ max-width: 600px; margin: 0 auto; padding: 20px; }}
                .header {{ background: linear-gradient(135deg, #7C3AED 0%, #6D28D9 100%); color: white; padding: 20px; border-radius: 10px 10px 0 0; }}
                .content {{ background: #f9f9f9; padding: 20px; border-radius: 0 0 10px 10px; }}
                .order-detail {{ background: white; padding: 15px; margin: 10px 0; border-radius: 5px; border-left: 4px solid #7C3AED; }}
                .label {{ font-weight: bold; color: #7C3AED; }}
                .footer {{ text-align: center; padding: 20px; color: #666; font-size: 12px; }}
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>🎂 New Cake Order!</h1>
                </div>
                <div class="content">
                    <p>A new cake order has been placed on CakeCamp.ng</p>
                    
                    <div class="order-detail">
                        <p><span class="label">Tracking Code:</span> <strong style="font-size: 18px; color: #7C3AED;">{order_data['tracking_code']}</strong></p>
                        <p><span class="label">Order ID:</span> {order_data['id']}</p>
                        <p><span class="label">Customer Name:</span> {order_data['customer_name']}</p>
                        <p><span class="label">Phone:</span> {order_data['phone']}</p>
                        {f'<p><span class="label">Email:</span> {order_data.get("email", "Not provided")}</p>' if order_data.get('email') else ''}
                        <p><span class="label">Delivery Address:</span> {order_data['delivery_address']}</p>
                        {f'<p><span class="label">Delivery Date:</span> {order_data.get("delivery_date", "Not specified")}</p>' if order_data.get('delivery_date') else ''}
                    </div>

                    <div class="order-detail">
                        <p><span class="label">Budget:</span> ₦{order_data['budget']:,.2f}</p>
                        <p><span class="label">Cake Size:</span> {order_data['cake_size'].capitalize()}</p>
                        <p><span class="label">Servings:</span> {order_data['servings']} people</p>
                    </div>

                    <div class="order-detail">
                        <p><span class="label">Design Type:</span> {order_data['design_type'].capitalize()}</p>
                        {f'<p><span class="label">Design Description:</span> {order_data.get("design_description", "")}</p>' if order_data.get('design_description') else ''}
                        {f'<p><span class="label">Cake Message:</span> {order_data.get("cake_message", "")}</p>' if order_data.get('cake_message') else ''}
                        {f'<p><span class="label">Special Requests:</span> {order_data.get("special_requests", "")}</p>' if order_data.get('special_requests') else ''}
                        {f'<p><span class="label">Packaging Requests:</span> {order_data.get("packaging_requests", "")}</p>' if order_data.get('packaging_requests') else ''}
                    </div>

                    <p style="margin-top: 20px; padding: 15px; background: #fff3cd; border-radius: 5px; border-left: 4px solid #ffc107;">
                        <strong>⏰ Action Required:</strong> Please review and confirm this order as soon as possible.
                    </p>
                </div>
                <div class="footer">
                    <p>CakeCamp.ng - Baking dreams into reality, one cake at a time.</p>
                    <p>Order placed at: {datetime.now(timezone.utc).strftime('%Y-%m-%d %H:%M:%S UTC')}</p>
                </div>
            </div>
        </body>
        </html>
        """

        # Attach HTML content
        html_part = MIMEText(html_body, 'html')
        msg.attach(html_part)

        # Create SSL context
        context = ssl.create_default_context()

        # Send email using SSL
        with smtplib.SMTP_SSL(SMTP_SERVER, SMTP_PORT, context=context) as server:
            server.login(SMTP_EMAIL, SMTP_PASSWORD)
            server.send_message(msg)
        
        logger.info(f"Order notification email sent successfully to {NOTIFICATION_EMAIL}")
        return True
    
    except Exception as e:
        logger.error(f"Failed to send email notification: {str(e)}")
        return False

# Define Models
class Order(BaseModel):
    model_config = ConfigDict(extra="ignore")
    
    id: str = Field(default_factory=lambda: str(uuid.uuid4()))
    tracking_code: str = Field(default_factory=generate_tracking_code)
    customer_name: str
    email: Optional[EmailStr] = None
    phone: str
    delivery_address: str
    delivery_date: Optional[str] = None
    budget: float
    cake_size: str
    servings: int
    design_type: str
    design_image: Optional[str] = None
    design_description: Optional[str] = None
    design_voice_note: Optional[str] = None
    cake_message: Optional[str] = None
    special_requests: Optional[str] = None
    packaging_requests: Optional[str] = None
    status: str = "pending"
    tracking_stages: List[dict] = Field(default_factory=lambda: [
        {"stage": "Order Received", "completed": True, "timestamp": datetime.now(timezone.utc).isoformat()},
        {"stage": "Order Confirmed", "completed": False, "timestamp": None},
        {"stage": "Baking in Progress", "completed": False, "timestamp": None},
        {"stage": "Quality Check", "completed": False, "timestamp": None},
        {"stage": "Ready for Delivery", "completed": False, "timestamp": None},
        {"stage": "Out for Delivery", "completed": False, "timestamp": None},
        {"stage": "Delivered", "completed": False, "timestamp": None}
    ])
    created_at: datetime = Field(default_factory=lambda: datetime.now(timezone.utc))
    updated_at: datetime = Field(default_factory=lambda: datetime.now(timezone.utc))

class OrderCreate(BaseModel):
    customer_name: str
    email: Optional[EmailStr] = None
    phone: str
    delivery_address: str
    delivery_date: Optional[str] = None
    budget: float
    cake_size: str
    servings: int
    design_type: str
    design_image: Optional[str] = None
    design_description: Optional[str] = None
    design_voice_note: Optional[str] = None
    cake_message: Optional[str] = None
    special_requests: Optional[str] = None
    packaging_requests: Optional[str] = None

class OrderUpdate(BaseModel):
    status: str

# Routes
@api_router.get("/")
async def root():
    return {"message": "Welcome to CakeCamp.ng API"}

@api_router.post("/orders", response_model=Order)
async def create_order(order_data: OrderCreate):
    try:
        order_dict = order_data.model_dump()
        order_obj = Order(**order_dict)
        
        doc = order_obj.model_dump()
        doc['created_at'] = doc['created_at'].isoformat()
        doc['updated_at'] = doc['updated_at'].isoformat()
        
        await db.orders.insert_one(doc)
        
        # Send email notification
        try:
            send_order_notification(order_obj.model_dump())
        except Exception as email_error:
            logger.error(f"Email notification failed but order was saved: {str(email_error)}")
            # Don't fail the order creation if email fails
        
        return order_obj
    except Exception as e:
        logger.error(f"Error creating order: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.get("/orders", response_model=List[Order])
async def get_orders():
    try:
        orders = await db.orders.find({}, {"_id": 0}).to_list(1000)
        
        for order in orders:
            if isinstance(order.get('created_at'), str):
                order['created_at'] = datetime.fromisoformat(order['created_at'])
            if isinstance(order.get('updated_at'), str):
                order['updated_at'] = datetime.fromisoformat(order['updated_at'])
        
        return orders
    except Exception as e:
        logger.error(f"Error fetching orders: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.get("/orders/{order_id}", response_model=Order)
async def get_order(order_id: str):
    try:
        order = await db.orders.find_one({"id": order_id}, {"_id": 0})
        
        if not order:
            raise HTTPException(status_code=404, detail="Order not found")
        
        if isinstance(order.get('created_at'), str):
            order['created_at'] = datetime.fromisoformat(order['created_at'])
        if isinstance(order.get('updated_at'), str):
            order['updated_at'] = datetime.fromisoformat(order['updated_at'])
        
        return order
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error fetching order: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.patch("/orders/{order_id}", response_model=Order)
async def update_order_status(order_id: str, update_data: OrderUpdate):
    try:
        order = await db.orders.find_one({"id": order_id}, {"_id": 0})
        
        if not order:
            raise HTTPException(status_code=404, detail="Order not found")
        
        update_dict = {
            "status": update_data.status,
            "updated_at": datetime.now(timezone.utc).isoformat()
        }
        
        await db.orders.update_one(
            {"id": order_id},
            {"$set": update_dict}
        )
        
        updated_order = await db.orders.find_one({"id": order_id}, {"_id": 0})
        
        if isinstance(updated_order.get('created_at'), str):
            updated_order['created_at'] = datetime.fromisoformat(updated_order['created_at'])
        if isinstance(updated_order.get('updated_at'), str):
            updated_order['updated_at'] = datetime.fromisoformat(updated_order['updated_at'])
        
        return updated_order
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error updating order: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.get("/orders/track/{order_id}", response_model=Order)
async def track_order(order_id: str):
    return await get_order(order_id)

@api_router.get("/orders/tracking/{tracking_code}", response_model=Order)
async def track_by_code(tracking_code: str):
    """Track order using tracking code"""
    try:
        order = await db.orders.find_one({"tracking_code": tracking_code.upper()}, {"_id": 0})
        
        if not order:
            raise HTTPException(status_code=404, detail="Order not found with this tracking code")
        
        if isinstance(order.get('created_at'), str):
            order['created_at'] = datetime.fromisoformat(order['created_at'])
        if isinstance(order.get('updated_at'), str):
            order['updated_at'] = datetime.fromisoformat(order['updated_at'])
        
        return order
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error tracking order: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.patch("/orders/{order_id}/tracking", response_model=Order)
async def update_tracking_stage(order_id: str, stage_index: int):
    """Update a specific tracking stage to completed"""
    try:
        order = await db.orders.find_one({"id": order_id}, {"_id": 0})
        
        if not order:
            raise HTTPException(status_code=404, detail="Order not found")
        
        tracking_stages = order.get('tracking_stages', [])
        
        if stage_index >= len(tracking_stages):
            raise HTTPException(status_code=400, detail="Invalid stage index")
        
        # Update the stage
        tracking_stages[stage_index]['completed'] = True
        tracking_stages[stage_index]['timestamp'] = datetime.now(timezone.utc).isoformat()
        
        # Update in database
        await db.orders.update_one(
            {"id": order_id},
            {"$set": {
                "tracking_stages": tracking_stages,
                "updated_at": datetime.now(timezone.utc).isoformat()
            }}
        )
        
        # Get updated order
        updated_order = await db.orders.find_one({"id": order_id}, {"_id": 0})
        
        if isinstance(updated_order.get('created_at'), str):
            updated_order['created_at'] = datetime.fromisoformat(updated_order['created_at'])
        if isinstance(updated_order.get('updated_at'), str):
            updated_order['updated_at'] = datetime.fromisoformat(updated_order['updated_at'])
        
        return updated_order
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error updating tracking stage: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

app.include_router(api_router)

app.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_origins=os.environ.get('CORS_ORIGINS', '*').split(','),
    allow_methods=["*"],
    allow_headers=["*"],
)

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

@app.on_event("shutdown")
async def shutdown_db_client():
    client.close()