import { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useAuth } from '@/context/AuthContext';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { toast } from 'sonner';
import { ArrowLeft, Calendar, Loader2 } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const CreateBooking = () => {
  const { providerId } = useParams();
  const navigate = useNavigate();
  const { user } = useAuth();
  const [provider, setProvider] = useState(null);
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    service_name: '',
    estimated_weight_kg: '',
    pickup_address: '',
    delivery_address: '',
    pickup_date: '',
    special_instructions: '',
    total_price: 0
  });

  useEffect(() => {
    fetchProvider();
  }, [providerId]);

  const fetchProvider = async () => {
    try {
      const response = await axios.get(`${API}/providers/${providerId}`);
      setProvider(response.data);
    } catch (error) {
      toast.error('Failed to load provider');
    }
  };

  const calculatePrice = (serviceName, weight) => {
    if (!provider || !serviceName || !weight) return 0;
    
    const service = provider.services.find(s => s.name === serviceName);
    if (!service) return 0;
    
    const basePrice = service.base_price;
    const weightPrice = service.price_per_kg ? service.price_per_kg * parseFloat(weight) : 0;
    
    return basePrice + weightPrice;
  };

  useEffect(() => {
    const price = calculatePrice(formData.service_name, formData.estimated_weight_kg);
    setFormData(prev => ({ ...prev, total_price: price }));
  }, [formData.service_name, formData.estimated_weight_kg]);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);

    try {
      const bookingData = {
        ...formData,
        provider_id: providerId,
        estimated_weight_kg: parseFloat(formData.estimated_weight_kg),
        total_price: formData.total_price,
        pickup_date: new Date(formData.pickup_date).toISOString()
      };

      await axios.post(`${API}/bookings`, bookingData);
      toast.success('Booking created successfully!');
      navigate('/my-bookings');
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Failed to create booking');
    } finally {
      setLoading(false);
    }
  };

  if (!provider) {
    return <div className="flex items-center justify-center min-h-screen">Loading...</div>;
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <nav className="bg-white border-b border-gray-200 px-6 py-4">
        <div className="max-w-7xl mx-auto flex items-center">
          <Button variant="ghost" onClick={() => navigate(-1)}>
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back
          </Button>
        </div>
      </nav>

      <div className="max-w-4xl mx-auto px-6 py-8">
        <div className="bg-white rounded-xl shadow-sm p-8">
          <h1 className="text-3xl font-bold mb-2">Book Service</h1>
          <p className="text-gray-600 mb-8">Booking with {provider.business_name}</p>

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Service Selection */}
            <div>
              <Label htmlFor="service_name">Select Service *</Label>
              <Select
                value={formData.service_name}
                onValueChange={(value) => setFormData({...formData, service_name: value})}
                required
              >
                <SelectTrigger data-testid="service-select" className="mt-2">
                  <SelectValue placeholder="Choose a service" />
                </SelectTrigger>
                <SelectContent>
                  {provider.services.map((service, idx) => (
                    <SelectItem key={idx} value={service.name}>
                      {service.name} - ₦{service.base_price}
                      {service.price_per_kg && ` + ₦${service.price_per_kg}/kg`}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Weight */}
            <div>
              <Label htmlFor="estimated_weight_kg">Estimated Weight (kg) *</Label>
              <Input
                data-testid="weight-input"
                id="estimated_weight_kg"
                type="number"
                step="0.1"
                value={formData.estimated_weight_kg}
                onChange={(e) => setFormData({...formData, estimated_weight_kg: e.target.value})}
                required
                className="mt-2"
              />
            </div>

            {/* Addresses */}
            <div>
              <Label htmlFor="pickup_address">Pickup Address *</Label>
              <Input
                data-testid="pickup-address-input"
                id="pickup_address"
                value={formData.pickup_address}
                onChange={(e) => setFormData({...formData, pickup_address: e.target.value})}
                required
                className="mt-2"
              />
            </div>

            <div>
              <Label htmlFor="delivery_address">Delivery Address *</Label>
              <Input
                data-testid="delivery-address-input"
                id="delivery_address"
                value={formData.delivery_address}
                onChange={(e) => setFormData({...formData, delivery_address: e.target.value})}
                required
                className="mt-2"
              />
            </div>

            {/* Pickup Date */}
            <div>
              <Label htmlFor="pickup_date">Pickup Date & Time *</Label>
              <Input
                data-testid="pickup-date-input"
                id="pickup_date"
                type="datetime-local"
                value={formData.pickup_date}
                onChange={(e) => setFormData({...formData, pickup_date: e.target.value})}
                required
                className="mt-2"
              />
            </div>

            {/* Special Instructions */}
            <div>
              <Label htmlFor="special_instructions">Special Instructions (Optional)</Label>
              <Textarea
                data-testid="instructions-textarea"
                id="special_instructions"
                value={formData.special_instructions}
                onChange={(e) => setFormData({...formData, special_instructions: e.target.value})}
                rows={3}
                className="mt-2"
              />
            </div>

            {/* Price Summary */}
            <div className="bg-blue-50 rounded-lg p-4">
              <div className="flex justify-between items-center">
                <span className="font-semibold">Estimated Total:</span>
                <span className="text-2xl font-bold text-blue-600">₦{formData.total_price.toFixed(2)}</span>
              </div>
              <p className="text-sm text-gray-600 mt-2">Final price may vary based on actual weight</p>
            </div>

            {/* Submit */}
            <div className="flex justify-end space-x-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => navigate(-1)}
              >
                Cancel
              </Button>
              <Button
                data-testid="confirm-booking-btn"
                type="submit"
                disabled={loading}
                className="gradient-blue text-white px-8"
              >
                {loading ? <><Loader2 className="w-5 h-5 mr-2 animate-spin" />Creating...</> : 'Confirm Booking'}
              </Button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
};

export default CreateBooking;